"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("./types");
const document_utils_1 = require("./document-utils");
const dummy_node_1 = require("./dummy/dummy-node");
const mutation_record_1 = require("./mutation/mutation-record");
const node_list_1 = require("./node-list");
const node_types_1 = require("./node-types");
const node_utils_1 = require("./node-utils");
const serialize_1 = require("./serializer/serialize");
const utils_1 = require("./utils");
/**
 * @see http://www.w3.org/TR/2000/REC-DOM-Level-2-Core-20001113/core.html#ID-1950641247
 */
class NodeImpl extends dummy_node_1.DummyNode {
    constructor() {
        super(...arguments);
        this.ELEMENT_NODE = node_types_1.NodeTypeTS.ELEMENT_NODE;
        this.ATTRIBUTE_NODE = node_types_1.NodeTypeTS.ATTRIBUTE_NODE;
        this.TEXT_NODE = node_types_1.NodeTypeTS.TEXT_NODE;
        this.CDATA_SECTION_NODE = node_types_1.NodeTypeTS.CDATA_SECTION_NODE;
        this.ENTITY_REFERENCE_NODE = node_types_1.NodeTypeTS.ENTITY_REFERENCE_NODE;
        this.ENTITY_NODE = node_types_1.NodeTypeTS.ENTITY_NODE;
        this.PROCESSING_INSTRUCTION_NODE = node_types_1.NodeTypeTS.PROCESSING_INSTRUCTION_NODE;
        this.COMMENT_NODE = node_types_1.NodeTypeTS.COMMENT_NODE;
        this.DOCUMENT_NODE = node_types_1.NodeTypeTS.DOCUMENT_NODE;
        this.DOCUMENT_TYPE_NODE = node_types_1.NodeTypeTS.DOCUMENT_TYPE_NODE;
        this.DOCUMENT_FRAGMENT_NODE = node_types_1.NodeTypeTS.DOCUMENT_FRAGMENT_NODE;
        this.NOTATION_NODE = node_types_1.NodeTypeTS.NOTATION_NODE;
        this.observers = [];
        this.firstChild = null;
        this.lastChild = null;
        this.previousSibling = null;
        this.nextSibling = null;
        this.parentNode = null;
        this.childNodes = null; // todo: use empty list instead of null maybe
        this.nodeValue = null;
        this.namespaceURI = null;
        this.prefix = null;
    }
    addObserver(observer, options) {
        for (const registered of this.observers) {
            if (registered.observer === observer) {
                // fixme: this is not according to the spec
                registered.options = options;
                return true;
            }
        }
        this.observers.push({ observer, options });
        return false;
    }
    delObserver(observer) {
        this.observers = this.observers.filter((registered) => registered.observer !== observer);
    }
    queueMutation(r) {
        const type = r.type;
        const name = r.attributeName;
        const namespace = r.attributeNamespace;
        const oldValue = r.oldValue;
        const target = r.target;
        const interestedObservers = new Map();
        const nodes = inclusiveAncestors(this);
        nodes.forEach((node) => {
            node.observers.forEach((registered) => {
                const options = registered.options;
                // https://dom.spec.whatwg.org/#queueing-a-mutation-record
                if (!(node !== target && options.subtree === false) &&
                    !(type === 'attributes' && options.attributes !== true) &&
                    !(type === 'attributes' &&
                        options.attributeFilter !== undefined &&
                        ((name != null && !options.attributeFilter.includes(name)) || namespace != null)) &&
                    !(type === 'characterData' && options.characterData !== true) &&
                    !(type === 'childList' && options.childList === false)) {
                    const mo = registered.observer;
                    if (!interestedObservers.has(mo)) {
                        interestedObservers.set(mo, null);
                    }
                    if ((type === 'attributes' && options.attributeOldValue === true) ||
                        (type === 'characterData' && options.characterDataOldValue === true)) {
                        interestedObservers.set(mo, oldValue);
                    }
                }
            });
        });
        interestedObservers.forEach((mappedOldValue, observer) => {
            const record = new mutation_record_1.MutationRecordImpl(r);
            record.oldValue = mappedOldValue;
            observer.queueRecord(record);
            // fixme: not entirely according to spec
            process.nextTick(() => observer.notify());
        });
    }
    // value: string | null = null; // todo: what is the purpose of this
    // ata: string | null = null;
    // Modified in DOM Level 2:
    insertBefore(newChild, refChild) {
        // raises
        const _newChild = document_utils_1._insertBefore(this, utils_1.asChildNode(newChild), refChild == null ? null : utils_1.asChildNode(refChild));
        // notify observers
        this.queueMutation({
            type: 'childList',
            target: this,
            addedNodes: new node_list_1.NodeListImpl(_newChild),
            removedNodes: new node_list_1.NodeListImpl(),
            previousSibling: _newChild.previousSibling,
            nextSibling: _newChild.nextSibling,
            attributeName: null,
            attributeNamespace: null,
            oldValue: null,
        });
        return _newChild;
    }
    replaceChild(newChild, oldChild) {
        // raises
        this.insertBefore(newChild, oldChild);
        return this.removeChild(oldChild);
    }
    removeChild(oldChild) {
        const _oldChild = document_utils_1._removeChild(this, oldChild);
        // notify observers
        this.queueMutation({
            type: 'childList',
            target: this,
            addedNodes: new node_list_1.NodeListImpl(),
            removedNodes: new node_list_1.NodeListImpl(_oldChild),
            previousSibling: _oldChild.previousSibling,
            nextSibling: _oldChild.nextSibling,
            attributeName: null,
            attributeNamespace: null,
            oldValue: null,
        });
        return _oldChild;
    }
    appendChild(newChild) {
        return this.insertBefore(newChild, null);
    }
    hasChildNodes() {
        return this.firstChild != null;
    }
    cloneNode(deep) {
        if (utils_1.isDocument(this)) {
            return node_utils_1.cloneNode(this, this, deep);
        }
        else {
            return node_utils_1.cloneNode(this.ownerDocument, this, deep);
        }
    }
    // Modified in DOM Level 2:
    normalize() {
        let child = this.firstChild;
        while (child) {
            const next = child.nextSibling;
            if (next && utils_1.isText(next) && utils_1.isText(child)) {
                this.removeChild(next);
                child.appendData(next.data);
            }
            else {
                child.normalize();
                child = next;
            }
        }
    }
    // Introduced in DOM Level 2:
    isSupported(feature, version) {
        if (utils_1.isDocument(this)) {
            return this.implementation.hasFeature(feature, version);
        }
        else {
            return this.ownerDocument.implementation.hasFeature(feature, version);
        }
    }
    // Introduced in DOM Level 2:
    hasAttributes() {
        return utils_1.isElement(this) ? this.attributes.length > 0 : false;
    }
    lookupPrefix(namespaceURI) {
        let el = this;
        while (el) {
            if (utils_1.isElement(el)) {
                const map = el._nsMap;
                // console.dir(map)
                if (map) {
                    for (const n in map) {
                        if (map[n] === namespaceURI) {
                            return n;
                        }
                    }
                }
            }
            el = utils_1.isAttr(el) ? el.ownerDocument : el.parentNode; // fixme: isn't this reversed ?
        }
        return null;
    }
    // Introduced in DOM Level 3:
    lookupNamespaceURI(prefix) {
        let el = this;
        while (el) {
            if (utils_1.isElement(el)) {
                const map = el._nsMap;
                // console.dir(map)
                if (map) {
                    if (prefix in map) {
                        return map[prefix];
                    }
                }
            }
            el = utils_1.isAttr(el) ? el.ownerDocument : el.parentNode; // fixme: isn't this reversed ?
        }
        return null;
    }
    // Introduced in DOM Level 3:
    isDefaultNamespace(namespaceURI) {
        const prefix = this.lookupPrefix(namespaceURI);
        return prefix == null;
    }
    toString(isHtml, nodeFilter) {
        const buf = [];
        const refNode = (utils_1.isDocument(this) && this.documentElement) || this;
        let prefix = refNode.prefix;
        const uri = refNode.namespaceURI;
        let visibleNamespaces;
        if (uri && prefix == null) {
            // console.log(prefix)
            prefix = refNode.lookupPrefix(uri);
            if (prefix == null) {
                // isHTML = true;
                visibleNamespaces = [
                    { namespace: uri, prefix: null },
                ];
            }
        }
        serialize_1.serializeToString(this, buf, isHtml, nodeFilter, visibleNamespaces);
        // console.log('###',this.nodeType,uri,prefix,buf.join(''))
        return buf.join('');
    }
    get textContent() {
        return node_utils_1.getTextContent(this);
    }
    set textContent(data) {
        if (utils_1.isElement(this) || utils_1.isDocumentFragment(this)) {
            while (this.firstChild) {
                this.removeChild(this.firstChild);
            }
            if (data) {
                this.appendChild(this.ownerDocument.createTextNode(data));
            }
        }
        else {
            // this.data = data;
            // this.value = data;
            this.nodeValue = data;
        }
    }
    ownerDocumentInternal() {
        if (utils_1.isDocument(this)) {
            return this;
        }
        else {
            return this.ownerDocument;
        }
    }
}
NodeImpl.ELEMENT_NODE = node_types_1.NodeTypeTS.ELEMENT_NODE;
NodeImpl.ATTRIBUTE_NODE = node_types_1.NodeTypeTS.ATTRIBUTE_NODE;
NodeImpl.TEXT_NODE = node_types_1.NodeTypeTS.TEXT_NODE;
NodeImpl.CDATA_SECTION_NODE = node_types_1.NodeTypeTS.CDATA_SECTION_NODE;
NodeImpl.ENTITY_REFERENCE_NODE = node_types_1.NodeTypeTS.ENTITY_REFERENCE_NODE;
NodeImpl.ENTITY_NODE = node_types_1.NodeTypeTS.ENTITY_NODE;
NodeImpl.PROCESSING_INSTRUCTION_NODE = node_types_1.NodeTypeTS.PROCESSING_INSTRUCTION_NODE;
NodeImpl.COMMENT_NODE = node_types_1.NodeTypeTS.COMMENT_NODE;
NodeImpl.DOCUMENT_NODE = node_types_1.NodeTypeTS.DOCUMENT_NODE;
NodeImpl.DOCUMENT_TYPE_NODE = node_types_1.NodeTypeTS.DOCUMENT_TYPE_NODE;
NodeImpl.DOCUMENT_FRAGMENT_NODE = node_types_1.NodeTypeTS.DOCUMENT_FRAGMENT_NODE;
NodeImpl.NOTATION_NODE = node_types_1.NodeTypeTS.NOTATION_NODE;
exports.NodeImpl = NodeImpl;
function inclusiveAncestors(node) {
    if (node.parentNode != null) {
        return [node, ...inclusiveAncestors(node.parentNode)];
    }
    else {
        return [node];
    }
}
//# sourceMappingURL=node.js.map