const moment = require('moment');
/**
 * Los método en la clase se van a referenciar a datos de tipo moment, 
 * lo que nos refereimos a esto, es al paquete moment disponible en npm
 */
export class Fecha {
    
    /**
     *  Convierte una fecha de formato String a una fecha tipo moment
     *
     *  <b>Autor: </b> Aldair Cruz
     *  <b>Fecha de Creación: </b> 12/03/2022
     * 
     * @param {string} fecha Se pasa la cadena de la fecha Ej: '2022-01-02'
     * @param {string} formato Se pasa el formato que se desea aplicar
     * @returns Regresa la fecha convertida a tipo moment con el formato deseado
     */
    static StoDate(fecha: string, formato: string) {
        return moment(fecha).format(formato);
    }

    /**
     * Obtiene la diferencia en dias entre 2 fechas especificas
     * 
     * <b>Autor: </b> Aldair Cruz
     * <b>Fecha de Creación: </b> 12/03/2022
     * 
     * @param {string} fecha_inicial Se pasa la fecha inicial
     * @param {string} fecha_final Se pasa la fecha final
     * @returns regresa la diferencia en dias entre la fecha inicial y la fecha final
     */
    static diferenciaDias(fecha_inicial: string, fecha_final: string) {
        return moment(fecha_final).diff(fecha_inicial, 'days');
    }

    /**
     * Obtiene la diferencia en meses entre 2 fechas especificas
     * 
     * <b>Autor: </b> Aldair Cruz
     * <b>Fecha de Creación: </b> 12/03/2022
     * 
     * @param {string} fecha_inicial Se pasa la fecha inicial
     * @param {string} fecha_final Se pasa la fecha final
     * @returns regresa la diferencia en meses entre la fecha inicial y la fecha final
     */
    static diferenciaMes(fecha_inicial: string, fecha_final: string) {
        return moment(fecha_final).diff(fecha_inicial, 'M');
    }

    /**
     * Obtiene la diferencia en años entre 2 fechas especificas
     * 
     * <b>Autor: </b> Aldair Cruz
     * <b>Fecha de Creación: </b> 12/03/2022
     * 
     * @param {string} fecha_inicial Se pasa la fecha inicial
     * @param {string} fecha_final Se pasa la fecha final
     * @returns regresa la diferencia en años entre la fecha inicial y la fecha final
     */
    static diferenciaAnios(fecha_inicial: string, fecha_final: string) {
        return moment(fecha_final).diff(fecha_inicial, 'years');
    }

    /**
     * Obtiene el día de la fecha proporcionada
     * 
     * @author Aldair Cruz
     * 
     * @param {string} fecha 
     * @returns 
     */
    static getDayOfMonth(fecha: string) {
        return Number(Fecha.StoDate(fecha, 'DD'));
    }

    /**
     * Obtiene el año de la fecha proporcionada
     * 
     * @author Aldair Cruz
     * 
     * @param {string} fecha 
     * @returns 
     */
     static getYear(fecha: string) {
        return Number(Fecha.StoDate(fecha, 'YYYY'));
    }

    /**
     * Obtiene el ultimo dia del mes que se le pase
     * 
     * @author Aldair Cruz
     * 
     * @param {string} fecha 
     * @returns 
     */
    static getUltimoDiaMes(fecha: string) {
        return this.StoDate(moment(fecha).endOf('month'), 'YYYY-MM-DD');
    }

    /**
     * Obtiene el dia que se le pase del mes y año proporcionado
     * 
     * @author Aldair Cruz
     * 
     * @param {string} fecha 
     * @param {int} dia 
     * @returns 
     */
    static withDayOfMonth(fecha: string, dia: number) {
        let mes = this.StoDate(fecha, 'MM');
        let anio = this.StoDate(fecha, 'YYYY');
        let fecha_dia = `${anio}-${mes}-${dia}`;
        return this.StoDate(fecha_dia, 'YYYY-MM-DD');
    }

    /**
     * Se le suma a la fecha proporcionada el número de meses deseado
     * 
     * @author Aldair Cruz
     * 
     * @param {string} fecha 
     * @param {int} numero_meses 
     * @returns 
     */
    static plusMonths(fecha: string, numero_meses: number) {
        return this.plusDate(fecha, numero_meses, 'M');
    }

    static minusMonths(fecha:string, numero_meses: number) {
        return this.minusDate(fecha, numero_meses, 'M');
    }

    /**
     * Se le suma a la fecha proporcionada la cantidad de dias deseados
     * 
     * @author Aldair Cruz
     * 
     * @param {string} fecha 
     * @param {int} numero_dias 
     * @returns 
     */
    static plusDays(fecha: string, numero_dias: number) {
        return this.plusDate(fecha, numero_dias, 'days');
    }


    static minusDays(fecha: string, numero_dias: number) {
        return this.minusDate(fecha, numero_dias, 'days');
    }

    static plusYears(fecha: string, numero_anios: number) {
        return this.plusDate(fecha, numero_anios, 'years');
    }

    static minusYears(fecha: string, numero_anios: number) {
        return this.minusDate(fecha, numero_anios, 'years');
    }

    static lengthOfMonth(fecha: string) {
        return this.StoDate(moment(fecha).daysInMonth(), 'DD');
    }

    /**
     * Obtiene el Dia de la Semana
     * 
     * @param {string} fecha 
     * @returns 
     */
    static getDayOfWeek(fecha: string) {
        return moment(fecha).day() + 1;
    }

    /**
     * Método estandar por el cual se pasa para poder realizar la suma de dia, mes, etc 
     * a la fecha deseada
     * 
     * @author Aldair Cruz
     * 
     * @param {string} fecha 
     * @param {int} numero 
     * @param {string} tipo 
     * @returns 
     */
    static plusDate(fecha: string, numero: number, tipo: string) {
        return this.StoDate(moment(fecha).add(numero, tipo), 'YYYY-MM-DD');
    }

    static minusDate(fecha: string, numero: number, tipo: string) {
        return this.StoDate(moment(fecha).subtract(numero, tipo), 'YYYY-MM-DD');
    }
}